## -*-Tcl-*- (nowrap)
 # ==========================================================================
 #  BibTeX mode - an extension package for Alpha
 # 
 #  FILE: "bibtexFile.tcl"
 #                                    created: 08/17/1994 {09:12:06 am} 
 #                                last update: 10/04/2001 {19:01:08 PM}
 #                                
 #  Description: 
 #  
 #  Menu item procedures that pertain to the whole file, as opposed to
 #  entry formatting, validation, and searching.
 #  
 #  See the "bibtexMode.tcl" file for license info, credits, etc.
 #  
 # ==========================================================================
 ## 

proc bibtexFile.tcl {} {}

# load main bib file!
bibtexMenu

namespace eval Bib {}

# ===========================================================================
# 
#  Application, File Lists  #
#        

proc Bib::bibtexApplication {} {
    global bibtexSig
    set name [app::launchAnyOfThese {BIBt Vbib CMTu} bibtexSig]
    switchTo [file tail $name]
}

proc Bib::bibtexHomePage {} {
    global BibmodeVars
    url::execute $BibmodeVars(homePage)
}

proc Bib::fileListProc {menuName item} {
    
    global Bib::TailFileConnect Bib::FileTails
    
    set item [string trimleft $item]
    
    if {$item == "Rebuild File List" || $item == "rebuildFileList"} {
        Bib::rebuildFileList
    } elseif {[lcontains Bib::FileTails $item]} {
        if {[catch {file::openQuietly [set Bib::TailFileConnect($item)]}]} {
            # The remaining items in the list are files to be opened.
            if {[askyesno "Could not open \"$item.\"\
              Would you like to rebuild the menu?"] == "yes"} {
                Bib::rebuildFileList
            } else {
                status::msg "Couldn't find \"$item\" ."
            } 
        }
    } else {
        Bib::$item
    } 
}

proc Bib::fileOptionsProc {menuName item} {
    
    global BibmodeVars Bib::PrefsInMenu2
    
    if {$menuName == "bibModeFiles" && [getModifiers]} {
        if {[lsearch -exact [set Bib::PrefsInMenu2] $item] != -1} {
	    cache::readContents index::prefshelp
	    if {[catch {set prefshelp(Bib,$item)} text]} {
		set text "Sorry, no information is available for '$item'"
	    } else {
		catch {unset index::prefshelp}
	    }
	    if {$BibmodeVars($item)} {set end "on"} else {set end "off"}
	    if {$end == "on"} {
		regsub {^.*\|\|} $text {} text
	    } else {
		regsub {\|\|.*$} $text {} text
	    }
	    append text "."
	    set msg "The '$item' preference for Bib mode is currently $end."
        } elseif {$item == "listAllBibliographies"} {
	    set text "Use this menu item to list all bibliographies\
              currently recognized by Bib mode.  This list is used\
              for creating databases and indices."
        } elseif {$item == "viewSearchPaths"} {
	    set text "Use this menu item to view the current search paths\
              associated with Bib mode."
        } elseif {$item == "addSearchPaths"} {
	    set text "Use this menu item to add additional search paths."
        } elseif {$item == "removeSearchPaths"} {
	    set text "Use this menu item to remove previously set search paths."
        } 
    } elseif {$item == "listAllBibliographies"} {
        Bib::listAllBibliographies "" 0
    } else {
        if {[lcontains Bib::PrefsInMenu2 $item]} {
            Bib::flagFlip $item
            if {$BibmodeVars($item)} {set end "on"} else {set end "off"}
            set msg "The \"$item\" preference is currently $end."
            if {$item == "buildFilesOnStart"} {return}
        } elseif {$item == "viewSearchPaths"} {
            mode::viewSearchPath ; return
        } elseif {$item == "addSearchPaths"} {
            mode::appendSearchPaths
        } elseif {$item == "removeSearchPaths"} {
            mode::removeSearchPaths
        }
        Bib::rebuildFileList
    }
    if {[info exists text]} {alertnote $text}
    if {[info exists msg]}  {status::msg $msg}
}

proc Bib::rebuildFileList {{quietly 0}} {

    Bib::listAllBibliographies
    Bib::rebuildMenu
    if {!$quietly} {Bib::reportFileCount}
}

# ===========================================================================
# 
# Bib::openDefaultBibFile
# 
# Open the user defined default .bib file.  If it has not been defined,
# this menu item should have been dimmed ...  The menu key binding is also
# valid in TeX mode, so long as this file has already been loaded ...  It
# would be nice if it were included in some latex<>.tcl file.
# 

Bind 0x1f <cs> {Bib::openDefaultBibFile} "TeX"

proc Bib::openDefaultBibFile {} {
    
    global Bib::DefaultFile
    
    if {[file exists [set Bib::DefaultFile]]} {
        file::openQuietly [set Bib::DefaultFile]
    } elseif {[dialog::yesno "The default .bib file could not be found.\
      Would you like to reset it?"]} {
        Bib::setDefaultBibFile
    } else {
        status::msg "Cancelled."
    } 
}

proc Bib::openAllBibFiles {} {
    
    foreach f [Bib::listAllBibliographies 1] {catch {file::openQuietly $f}}
}

proc Bib::closeAllBibFiles {} {
    
    foreach f [Bib::listAllBibliographies 1] {
        if {[lsearch [winNames -f] $f] != "-1"} {
            bringToFront [file tail $f]
            catch {killWindow}
        } 
    }
}

# ===========================================================================
# 
#  Cite Key Lists  #
# 
# Note: I realize that some of this is not so efficient ...  The check for
# duplicate cite-keys within a file is fine, as is the multiple file checks. 
# Retreiving the list of cite-keys isn't so great.  It would be better to
# somehow retrieve the info from bibIndex.  -- cbu
# 

proc Bib::citeKeysProc {menuName item} {
    
    global Bib::CiteKeys Bib::TailFileConnect2
    
    set fileTails ""
    if {[llength [set Bib::CiteKeys]]} {
	set fileTails [lindex [set Bib::CiteKeys] 2]
    } 
    if {$item == "countEntries"} {
        Bib::countEntries
    } elseif {$item == "findDuplicates"} {
        Bib::findDuplicates
    } elseif {$item == "findAllDuplicates"} {
        Bib::findAllDuplicates
    } elseif {$item == "listCiteKeys"} {
        Bib::listCiteKeys
    } elseif {$item == "createCiteKeyList"} {
        set title "Select the files to include:"
        set first [list "Include all files"]
        Bib::listAllCiteKeys "" $title $first 1
    } elseif {$item == "clearCiteKeyList"} {
        Bib::clearCiteKeyList
    } elseif {[lsearch $fileTails $item] != "-1"} {
        # This works :
        # 
        # Bib::listAllCiteKeys $item
        # 
        # but it's not very efficient, especially when there are a lot of
        # cite-keys listed in Bib::CiteKeys.  But we'll still give them
        # something ...
        Bib::listCiteKeys [set Bib::TailFileConnect2($item)]
    } else {
        Bib::$item
    } 
}

# ===========================================================================
# 
# Report the number of entries of each type
# 

proc Bib::countEntries {{quietly 0}} {

    Bib::BibModeMenuItem
    
    global Bib::TopPat1 Bib::TopPat2
    
    set pos [minPos]
    set pat [set Bib::TopPat1]
    set count 0
    catch {unset type}
    
    set lines    "\r\"[win::CurrentTail]\"\r\r"
    while {![catch {search -f 1 -r 1 -m 0 -i 0 -s $pat $pos} result]} {
        status::msg "Counting entries  [incr count]"
        set start [lindex $result 0]
        set end   [nextLineStart $start]
        set text  [getText $start $end]
        set lab ""
        if {[regexp [set Bib::TopPat2] $text match entryType]} {
            set entryType [string tolower $entryType]
            if {[catch {incr type($entryType)} num]} {
                set type($entryType) 1
            }
        }
        set pos $end
    }
    foreach name [lsort [array names type]] {
        if {$type($name) > 0} {
            append lines [format "%4.0d  %s\n" $type($name) $name]
        }
    }
    append lines "----  -----------------\n"
    append lines [format "%4.0d  %s\n" $count "Total entries"]
    if {!$quietly} {
        set t    "% -*-Bib-*- (stats)\r"
        append t $lines
        new -n {* BibTeX Statistics *} -m "Bib" -text $t
        winReadOnly
        catch {shrinkWindow 1}
    } else {
        return [list $count $lines]
    } 
}

proc Bib::countAllEntries {{fileList ""}} {
    
    global Bib::FileTails Bib::TailFileConnect2
    
    if {$fileList == ""} {
        set title "Choose some files to count :"
        set fileList [listpick -l -p $title [set Bib::FileTails]]
    } 
    set currentTails [winNames]
    set count 0
    set t ""
    foreach fileTail $fileList {
        file::openQuietly [set Bib::TailFileConnect2([lindex $fileTail 0])]
        set result [Bib::countEntries 1]
        incr count [lindex $result 0]
        append t   [lindex $result 1]
        if {[lsearch $currentTails [win::CurrentTail]] == "-1"} {killWindow} 
    }
    set t1    "% -*-Bib-*- (stats)\r"
    append t1 $t
    append t1 "\r[format "%4.0d  %s\n" $count "Total entries, all files"]"
    new -n {* BibTeX Statistics *} -m "Bib" -text $t1
    winReadOnly
    catch {shrinkWindow 1}
}

# ===========================================================================
# 
# Report all duplicate cite keys for the current window, or across several
# files.  We save all of the cite-key information in a "Bib::CiteKeys"
# variable that can be used for subsequent duplicate cite-key queries, or
# unset by the user.  When the mode is first loaded, this variable is set
# empty.
#
    
proc Bib::findDuplicates {{compareFiles ""} {wCT ""} {quietly 0}} {
    
    global Bib::CiteKeys
    
    set currentTail [win::CurrentTail]
    if {$compareFiles == ""} {
        if {[llength [set Bib::CiteKeys]] && [askyesno \
          "Do you want to compare this file with the cite-keys\
          currently stored? (If not, that list will be deleted.)"] == "yes"} {
            set compareFiles 1
        } else {
            set compareFiles 0
        }
    } 
    if {!$compareFiles} {
        set citeKeys     ""
        set lineNumbers  ""
        set fileTails    ""
    } else {
        set citeKeys     [lindex [set Bib::CiteKeys] 0]
        set lineNumbers  [lindex [set Bib::CiteKeys] 1]
        set fileTails    [lindex [set Bib::CiteKeys] 2]
        set priorTails   [lunique $fileTails]
        if {[lsearch $priorTails $currentTail] != "-1"} {
            alertnote "The cite-keys from this window have already been\
              included in the list of citations.  If this file has since\
              changed, you should \"Clear Cite Key List\" and start fresh."
            status::msg "Cancelled."
            error "The current window has already been included in Bib::CiteKeys."
        } 
    } 
    set dupCount 0
    set count    0
    set duplicates  ""
    set citeKeyList ""
    set results [Bib::MarkFile 1]
    if {!$quietly} {
        status::msg "Checking for duplicate cite-keys "
    }
    foreach result $results {
        set citeKey     [lindex $result 0]
        set lineNumber  [lindex $result 1]
        set fileTail    [lindex $result 2]
        incr count
        if {$quietly} {
            # We want the entire list of citeKeys
            append citeKeyList "[format {%-30s} $citeKey]"
            append citeKeyList "[format {%4d} $lineNumber]"
            append citeKeyList "    \"$currentTail\"\r"
        }
        set indexNumber [lsearch $citeKeys $citeKey]
        if {$indexNumber != "-1"} {
            set priorCiteKey [lindex $citeKeys    $indexNumber]
            set priorLine    [lindex $lineNumbers $indexNumber]
            set priorTail    [lindex $fileTails   $indexNumber]
            append duplicates "[format {%-30s} $priorCiteKey]"
            append duplicates "[format {%4d} $priorLine]"
            append duplicates "    \"$priorTail\"\r"
            append duplicates "[format {%-30s} $citeKey]"
            append duplicates "[format {%4d} $lineNumber]"
            append duplicates "    \"$currentTail\"\r\r"
            incr dupCount
        }
        # We're creating the list at the same time.
        lappend citeKeys    $citeKey
        lappend lineNumbers $lineNumber
        lappend fileTails   $currentTail

    }
    # Set these for the next round.
    set Bib::CiteKeys [list $citeKeys $lineNumbers $fileTails]
    set entryLength [llength $results]
    # If "quietly", all we want to do is return the count and the list.
    if {$quietly} {return [list $count $citeKeyList]} 
    set appendResults 0
    if {$wCT != ""} {set appendResults $wCT} 
    if {!$dupCount && $appendResults == 0} {
        status::msg "$count entries checked -- no duplicate cite-keys."
        Bib::rebuildMenu "citeKeyLists"
        return
    } 
    set duplicateWindows ""
    foreach window [winNames] {
        if {[regexp {\\* Cite\-Keys Results \\*} $window]} {
            lappend duplicateWindows $window
        } 
    }
    set duplicateWindows 
    if {$appendResults == 0 && [llength $duplicateWindows]} {
        if {[askyesno "Would you like to append these results to your\
          current Results window"] == "yes"} {
            if {[llength $duplicateWindows] != 1} {
                set appendResults [listpick -p "Please choose a window:" \
                  $duplicateWindows]
            } else {
                set appendResults [lindex $duplicateWindows 0]
            } 
        } 
    } 
    # Generate the report.
    set    t "\rDuplicate Cite-Key Results for \"[win::CurrentTail]\"\r\r"
    append t "Note: Command double-click on any line-number or cite-key\r"
    append t "      to return to its entry.\r"
    append t "_________________________________________________________\r"
    if {$compareFiles} {
        append t "\r        Files Compared: "
        foreach window [lunique $priorTails] {
            append t "\"$window\"\r                        "
        }
    } 
    append t "\r  Entries in this file:  [format {%4d} $count]\r"
    if {$compareFiles} {
        append t "         Total Entries:  [format {%4d} [llength $fileTails]]\r"
    } 
    append t "   Duplicate Cite-keys:  [format {%4d} $dupCount]\r"
    append t "_________________________________________________________\r"
    if {$dupCount} {
        append t "\rcite-keys:                   line #:  file name:\r"
        append t "----------                   -------  ----------\r\r"
        append t $duplicates
        append t "_________________________________________________________\r"
    } 
    append t "_________________________________________________________\r"
    status::msg "$dupCount duplicate cite-keys found."
    # Either create a new window, or append to an existing one.
    if {$appendResults == 0 || $appendResults == "-1"} {
         set t1 "% -*-Bib-*- (cite-keys)\r"
         append t1 $t
         new -n "* Cite-Keys Results *" -m "Bib" -text $t1
         set pos [minPos]
    } else {
        bringToFront $appendResults
        setWinInfo read-only 0
        goto [maxPos]
        set pos [getPos]
        insertText $t
    } 
    winReadOnly
    goto $pos ; insertToTop
    Bib::rebuildMenu "citeKeyLists"
    return $dupCount
}

proc Bib::findAllDuplicates {{biblist ""}} {

    global Bib::FileTails Bib::TailFileConnect Bib::CiteKeys
    
    # Get the names of current windows, which we won't close
    set currentTails [winNames]
    # Get the list of files to check.
    if {$biblist == ""} {
        set bibfiles {"Check all files "}
        foreach f [set Bib::FileTails] {lappend bibfiles $f} 
        set title "Select the files to search for duplicates :"
        set biblist [listpick -l -L {"Check all files "} -p $title $bibfiles]
    } 
    if {[regexp "Check all files " $biblist]} {set biblist [set Bib::FileTails]} 
    # Now check the first one, flushing the Bib::CiteKeys variable,
    # and creating a new report.
    set firstFile [lindex $biblist 0]
    file::openQuietly [set Bib::TailFileConnect($firstFile)]
    set currentTail [win::CurrentTail]
    set results [Bib::findDuplicates 0 "-1" 0]
    if {[lsearch $currentTails $currentTail] == "-1"} {
        bringToFront $currentTail
        killWindow
    } 
    # Remove that from the list
    set biblist [lreplace $biblist 0 0]
    # The current window is now "Cite-Keys Results"
    set wCT [win::CurrentTail]
    foreach f $biblist {
        set f [lindex $biblist 0]
        file::openQuietly [set Bib::TailFileConnect($f)]
        set currentTail [win::CurrentTail]
        incr results [Bib::findDuplicates 1 $wCT 0]
        if {[lsearch $currentTails $currentTail] == "-1"} {
            bringToFront $currentTail
            killWindow
        } 
        # Remove that from the list
        set biblist [lreplace $biblist 0 0]
    } 
    goto [minPos]
    Bib::rebuildMenu "citeKeyLists"
    status::msg "$results duplicate cite-keys found."
}

# ===========================================================================
# 
# List Cite Keys (formerly Index This Window)
# 
# Collect the citekeys and titles of all entries in the current window, and
# return the results in a new window.  Command double clicking will allow the
# user to jump to the original citation.  Bib::makeDatabaseOf will now format
# the results nice if given the 1 flag ...
# 

proc Bib::listCiteKeys {{fullPath ""}} {
    
    global Bib::TailFileConnect2
    
    set currentTails [winNames]
    if {$fullPath == ""} {
        set fileTail [win::Current]
    } else {
        file::openQuietly $fullPath
    }
    set currentTail [win::CurrentTail]
    set results [Bib::makeDatabaseOf $fullPath 0]
    set t    "% -*-Bib-*- (index)\r"
    append t "\rIndex for \"[win::CurrentTail]\"\r\r"
    append t "Note: This is NOT the bibIndex used for \"Quick Find Citation\"\r"
    append t "      or for TeX mode electric citation completions ...\r\r"
    append t "      Command double-click on any cite-key to return to its entry.\r"
    append t "__________________________________________________________________\r\r"
    append t $results
    new -n "* Cite-Keys for [file tail $currentTail] *" -m "Bib" -text $t
    winReadOnly
    Bib::MarkFile
    status::msg "Entries are listed in the marks menu."
}


proc Bib::listAllCiteKeys {{fileName ""} {title ""} {first ""} {quietly 0}} {
    
    global Bib::CiteKeys Bib::FileTails Bib::TailFileConnect
    
    if {$title == ""} {set title "Select the files to search for duplicates:"}
    if {$first == ""} {set first [list "Check for duplicates in all files "]}
    set create 0
    if {$quietly} {set create 1} 
    if {![llength [set Bib::CiteKeys]] && !$quietly} {
        alertnote "There are no current cite-keys saved.\
          Please choose some files that you want to list."
        set title "Select the files to list:"
        set first [list "List cite-keys for all files "]
        set create 1
    }
    if {$create} {
        # Get the names of current windows, which we won't close
        set currentTails [winNames]
        set bibfiles $first
        foreach f [set Bib::FileTails] {lappend bibfiles $f} 
        set biblist [listpick -l -p $title $bibfiles]
        if {[regexp $first $biblist]} {set biblist [set Bib::FileTails]}
        set Bib::CiteKeys ""
        set t           ""
        foreach f $biblist {
            file::openQuietly [set Bib::TailFileConnect($f)]
            set currentTail [win::CurrentTail]
            set results  [Bib::findDuplicates 1 0 1]
            if {[lsearch $currentTails $currentTail] == "-1"} {
                bringToFront $currentTail
                killWindow
            } 
            set count    [lindex $results 0]
            set citeKeys [lindex $results 1]
            append t "\rCite-Key List for \"[win::CurrentTail]\"\r\r"
            append t "Note: Command double-click on any line-number or cite-key\r"
            append t "      to return to its entry.\r"
            append t "\r  Entries in this file:  [format {%4d} $count]\r"
            append t "_________________________________________________________\r\r"
            append t "cite-keys:                   line #:  file name:\r"
            append t "----------                   -------  ----------\r\r"
            append t $citeKeys
            append t "_________________________________________________________\r"
            append t "_________________________________________________________\r"
        } 
    } else {
        # We're dealing with a list that has already been created.
        set citeKeys     [lindex [set Bib::CiteKeys] 0]
        set lineNumbers  [lindex [set Bib::CiteKeys] 1]
        set fileTails    [lindex [set Bib::CiteKeys] 2]
        set t ""
        set lastfileTail ""
        set oneFileOnly  0
        # If this was called for a specific file, we only list those cite-keys.
        if {$fileName != ""} {set oneFileOnly 1} 
        for {set indexNumber 0} {$indexNumber < [llength $citeKeys]} {incr indexNumber} {
            # All we want is the list of citeKeys
            set citeKey        [lindex $citeKeys    $indexNumber]
            set lineNumber     [lindex $lineNumbers $indexNumber]
            set fileTail       [lindex $fileTails   $indexNumber]
            if {!$oneFileOnly} {set fileName $fileTail} 
            if {$fileTail != $lastfileTail && $fileTail == $fileName} {
                status::msg "Creating the cite-key list for \"$fileTail\" "
                append t "\rCite-Key List for \"$fileTail\"\r\r"
                append t "Note: Command double-click on any line-number or cite-key\r"
                append t "      to return to its entry.\r\r"
                append t "_________________________________________________________\r\r"
                append t "cite-keys:                   line #:  file name:\r"
                append t "----------                   -------  ----------\r\r"
            } 
            if {$fileTail == $fileName} {
                append t "[format {%-30s} $citeKey]"
                append t "[format {%4d} $lineNumber]"
                append t "    \"$fileTail\"\r"
                set lastfileTail $fileTail
            } 
        }
    }
    if {!$quietly} {
        set t1 "% -*-Bib-*- (cite-keys)\r"
        append t1 $t
        new -n "* Cite-Keys List *" -m "Bib" -text $t1
        winReadOnly
    } else {
        status::msg "The new list of citations has been created."
    } 
    Bib::rebuildMenu "citeKeyLists"
}

proc Bib::clearCiteKeyList {} {
    
    global Bib::CiteKeys
    if {[llength [set Bib::CiteKeys]]} {
        set Bib::CiteKeys ""
        status::msg "The list of cite-keys has been cleared."
    } else {
        status::msg "The list of cite-keys is currently empty."
    } 
    Bib::rebuildMenu "citeKeyLists"
    
}

# ===========================================================================
# 
#  Sorting, Marking   #
#
    
# ===========================================================================
# 
# Sort the file by various criteria.
# 

proc Bib::sortFileByProc {menuName item} {
    
    global BibmodeVars
    
    status::msg "Sorting '[win::CurrentTail]' ..."
    
    switch $item {
        "citeKey"               {Bib::sortByCiteKey    }
        "firstAuthor,Year"      {Bib::sortByAuthors 0 0}
        "lastAuthor,Year"       {Bib::sortByAuthors 1 0}
        "year,FirstAuthor"      {Bib::sortByAuthors 0 1}
        "year,FirstAuthor"      {Bib::sortByAuthors 1 0}
    }
    status::msg "'[win::CurrentTail]' has been sorted."
    if {$BibmodeVars(autoMark)} {Bib::MarkFile} 
}

# ===========================================================================
# 
# Sorting Preliminaries
# 

# ===========================================================================
# 
# Return a list of the cite-keys of all cross-referenced entries.
#

proc Bib::listCrossrefs {} {
    set matches [Bib::findEntries {crossref}]
    catch {unset crossrefs}
    
    status::msg "scanning for crossrefs"
    foreach hit $matches {
        set top [lindex $hit 2] 
        set bottom [lindex $hit 3]
        set entry [getText $top $bottom]
        regsub -all "\[\n\r\]+" $entry { } entry
        regsub -all "\[     \]\[    \]+" $entry { } entry
        regsub {[,  ]*[\)\}][   ]*$} $entry { } entry
        if {![catch {Bib::getFldValue $entry crossref} fldval]} {
            set fldval [string tolower $fldval]
            if {[catch {incr crossref($fldval)} num]} {set crossrefs($fldval) 1}
        }
    }
    if {[catch {lsort [array names crossrefs]} res]} {set res {}}
    status::msg ""
    return $res
}

# ===========================================================================
# 
# Extract the data from the indicated field of an entry, which is passed as
# a single string.  This version tries to be completely general, allowing
# nested braces within data fields and ignoring escaped delimiters. 
# (derived from proc getField).
#

proc Bib::getFldValue {entry fldname} {

    set fldPat  "\[\t  \]*${fldname}\[\t \]*=\[\t \]*"
    set fldPat2 {,[\t ]*([^ =,]+)[\t ]*=[\t ]*}
    set slash   "\\"
    set qslash  "\\\\"
    
    set ok [regexp -indices -nocase $fldPat $entry match]
    if {$ok} {
        set pos [expr [lindex $match 1] + 1]
        set entry [string range $entry $pos end]
        
        if {[regexp -indices $fldPat2 $entry match sub1]} {
            set entry [string range $entry 0 [expr [lindex $match 0]-1]]
        } 
        return [Bib::bibFieldData $entry]
    } else {
        error "field not found"
    }
}

# ===========================================================================
# 
# Sort all of the entries in the file alphabetically by their cite-keys.
#

proc Bib::sortByCiteKey {} {
    
    global Bib::TopPat Bib::TopPat1 BibmodeVars
    
    set bibSegStr $BibmodeVars(segregateStrings)
    set matches   [Bib::findEntries [set Bib::TopPat]]
    set crossrefs [Bib::listCrossrefs]
    set strings   [Bib::listStrings]
    
    set begEntries [maxPos]
    set endEntries [minPos]
    
    set strs {}
    set vals {}
    set refs {}
    
    foreach hit $matches {
        set beg [lindex $hit 0]
        set end [lindex $hit 1]
        set top [lindex $hit 2] 
        set bottom [lindex $hit 3]
        if {[regexp [set Bib::TopPat1] [getText $top $bottom] allofit citeKey]} {
            set citeKey [string tolower $citeKey]
            set keyExists 1
        } else {
            set citeKey "000000$beg"
            set keyExists 0
        }
        if {$keyExists && [lsearch -exact $crossrefs $citeKey] >= 0} {
            lappend refs [list $top $top $bottom]
        } elseif {$keyExists && $bibSegStr && [lsearch -exact $strings $citeKey] >= 0} {
            lappend strs [list $citeKey $top $bottom]       
        } else {
            lappend vals [list $citeKey $top $bottom]
        }
        
        if {[pos::compare $top    < $begEntries]} {set begEntries $top}
        if {[pos::compare $bottom > $endEntries]} {set endEntries $bottom}
    }
    
    if {$bibSegStr} {
        set result [concat $strs [lsort $vals] $refs]
    } else {
        set result [concat [lsort $vals] $refs]
    }
    
    if {[llength $result] > 0} {
        Bib::writeEntries $result 1 $begEntries $endEntries 1
    } else {
        status::msg "No results of cite-key sort !!??"
    }
}

# ===========================================================================
# 
# Sort all of the entries in the file alphabetically by author.
#

proc Bib::sortByAuthors {{lastAuthorFirst 0} {yearFirst 0}} {
    
    global Bib::TopPat Bib::TopPat1 BibmodeVars
    set bibSegStr $BibmodeVars(segregateStrings)
    
    set matches   [Bib::findEntries [set Bib::TopPat]]
    set crossrefs [Bib::listCrossrefs]
    set strings   [Bib::listStrings]
    
    set vals {}
    set others {}
    set refs {}
    set strs {}
    
    set beg [maxPos]
    set end [minPos]
    
    foreach hit $matches {
        set pos [lindex $hit 1]
        set top [lindex $hit 2] 
        set bottom [lindex $hit 3]
        set entry [getText $top $bottom]
        regsub -all "\[\n\r\]+" $entry { } entry
        regsub -all "\[     \]\[    \]+" $entry { } entry
        regsub {[,  ]*[\)\}][   ]*$} $entry { } entry
        if {[regexp [set Bib::TopPat1] $entry allofit citeKey]} {
            set citeKey [string tolower $citeKey]
            set keyExists 1
        } else {
            set citeKey ""
            set keyExists 0
        }
        
        if {$keyExists && [lsearch -exact $crossrefs $citeKey] >= 0} {
            lappend refs [list $pos $top $bottom]
        } elseif {$bibSegStr && $keyExists && [lsearch -exact $strings $citeKey] >= 0} {
            lappend strs [list $citeKey $top $bottom]       
        } else {
            if {![catch {Bib::getFldValue $entry author} fldval]} {
                if {[catch {Bib::getFldValue $entry year} year]} { set year 9999 }
                lappend vals [list [Bib::authSortKey $fldval $lastAuthorFirst $year $yearFirst] $top $bottom]
            } else {
                lappend others [list $pos $top $bottom]
            }
        }
        if {[pos::compare $top    < $beg]} {set beg $top}
        if {[pos::compare $bottom > $end]} {set end $bottom}
    }
    
    if {$bibSegStr} {
        set result [concat $strs $others [lsort $vals] $refs]
    } else {
        set result [concat $others [lsort $vals] $refs]
    }
    
    if {[llength $result] > 0} {
        Bib::writeEntries $result 1 $beg $end 1
    } else {
        status::msg "No results of author sort !!??"
    }
}

# ===========================================================================
# 
# Create a sort key from an author list.  When sorting entries by author,
# performing the sort using keys should be faster than reparsing the author
# lists for every comparison (the old method :-( ).
#

proc Bib::authSortKey {authList lastAuthorFirst {year {}} {yearFirst 0}} {
    global BibmodeVars
    set pat1 {\\.\{([A-Za-z])\}}
    set pat2 {\{([^\{\}]+) ([^\{\}]+)\}}
    
    # Remove enclosing braces, quotes, or whitespace
    set auths %[string trim $authList {{}"  }]&
    # Remove TeX codes for accented characters
    regsub -all -- $pat1 $auths {\1} auths
    # Concatenate strings enclosed in braces
    while {[regsub -all $pat2 $auths {{\1\2}} auths]} {}
    # Remove braces (curly and square)
    regsub -all {[][\{\}]} $auths {} auths
    #   regsub -all {,} $auths { ,} auths
    # Replace 'and's with begin-name/end-name delimiters
    regsub -all {[  ]and[   ]} $auths { \&% } auths
    # Put last name first in name fields without commas
    regsub -all {%([^\&,]+) ([^\&, ]+) *\&} $auths {%\2,\1\&} auths
    # Remove begin-name delimiters
    regsub -all {%} $auths {} auths
    # Remove whitespace surrounding name separators
    regsub -all {[  ]*\&[   ]*} $auths {\&} auths
    # Replace whitespace separating words with shrieks 
    regsub -all {[  ,]+} $auths {!} auths
    # If desired, move last author to head of sort key
    if {$lastAuthorFirst} {
        regsub {(.*)&([^&]+)&?$} $auths {\2\&\1} auths
    }
    # If provided, sort by year (descending order) as well
    regsub {^[^0-9]*([0-9]*).*$} $year {\1} year
    if {$year != {}} {
        if {$BibmodeVars(descendingYears)} { catch {set year [expr 9999-$year]} }
        if {$yearFirst} {
            set auths "$year&$auths"
        } else {        
            regsub {^([^&]+)(&?)} $auths "\\1\\&${year}\\2" auths
        }
    }
    
    return $auths
}

# ===========================================================================
# 
# Sort the file marks.
# 
# (These operations are also available under the "Search:NamedMarks" menu).
#

proc Bib::sortMarksProc {menuName item} {
    
    Bib::BibModeMenuItem
    
    if {$item == "alphabetically"} {
        sortMarksFile
    } elseif  {$item == "byPosition"} {
        orderMarks
    }
}

# ===========================================================================
# 
# Conversions
# 
# All conversion extensions / features should now name their procedures
# 
# "Bib::menuItem"
# 
# as in Bib::bibToHtml, Bib::bibToRefer etc.  and should insert themselves
# into the "bibtexConversions" menu.
#
        
# ===========================================================================
# 
#  Default Entries, Fields  #
#

proc Bib::entryFieldsProc {menuName item} {
    
    global BibmodeVars
    
    if {$item == "restoreDefaultFields"} {
        Bib::restoreDefaultFields
    } elseif {$item == "removeCustomEntry"} {
        Bib::removeCustomEntry
    } elseif {[regexp {CustomField} $item]} {
        if {[llength [winNames]] && [Bib::isBibFile "" 1] && [askyesno \
          "Would you like to add all of the \"extra\" fields from this window\
          to the \"Add Fields\" preference?"] == "yes"} {
            Bib::addWindowFields
        } else {
            Bib::editPreference "addFields" "Edit the \"Add Fields\" preference:" 1
        }
        Bib::updatePreferences addFields
    } else {
        Bib::editEntryFields $item
    } 
}


# ===========================================================================
# 
# Edit Custom Entry
# 
# Change the custom<EntryName> field preferences, creating them if
# necessary, or (if the new field list is empty) optionally restore them to
# defaults by removing the preference.  User defined entries can be removed
# entirely if the new field list is empty, but can also remain.
# 
# The name should be names of the form "book", not "customBook".
# 

proc Bib::editEntryFields {{entryName ""}} {
    
    global BibmodeVars Bib::Entries Bib::RqdFlds Bib::MyFlds
    
    if {$entryName == ""} {
        set entryName [listpick -p \
          "Select an entry to edit:" [set Bib::Entries]]
    }
    if {$entryName == "string" || $entryName == "customEntry" || \
      [info exists Bib::MyFlds($entryName]} {
        status::msg "Sorry, the entry \"$entryName\" cannot be edited from this menu."
        return
    } 
    # If by some chance the entryName is both not in the Bib::RqdFlds() array
    # and doesn't have a preference, we should try rebuilding the menu.
    if {[lsearch [set Bib::Entries] $entryName] == "-1"} {
        alertnote "Couldn't find any information for \"$entryName\".\
          It will be removed from the menu."
        Bib::updatePreferences customEntry
        return
    }
    # Find the custom pref's name, and if it exists use its fields.
    # Otherwise, use the Bib::RqdFlds()'s fields.
    set customEntryName [Bib::entryPrefConnect $entryName]
    if {[info exists BibmodeVars($customEntryName)]} {
        set defaultList $BibmodeVars($customEntryName)
    } else {
        set defaultList [set Bib::RqdFlds($entryName)]
    }
    # Get the new list of default fields.
    set newEntryFields [getline "Modify the list of fields\
      for the entry \"$entryName\":" $defaultList]
    if {$newEntryFields == ""} {status::msg "Cancelled." ; return}
    if {$newEntryFields == $defaultList} {
        status::msg "Nothing was entered -- \"$entryName\" fields are unchanged."
    } else {
        # Update the mode preferences, creating the new pref if necessary.
        set BibmodeVars($customEntryName) $newEntryFields
	prefs::modified BibmodeVars($customEntryName)
    }
    Bib::updatePreferences customEntry
    status::msg "Current \"$entryName\" fields: $BibmodeVars($customEntryName)"
}

# ===========================================================================
# 
# Restore Default Fields.
# 
# Restore the fields of an entry to those defined in Bib::RqdFlds().  This will
# remove any custom<EntryName> preferences.  If the entry is a custom entry
# defined by the user (via the "Custom Entry" menu item), the user is
# instead given the option to remove it, although these entries are not
# included in the listpick.
# 
# The list should be names of the form "book", not "customBook".
# 

proc Bib::restoreDefaultFields {{entryNameList ""}} {
    
    global BibmodeVars Bib::CustomEntryList2
    
    if {$entryNameList == ""} {
        set title "Select entries to restore :"
        set entryNameList [listpick -l -p $title [set Bib::CustomEntryList2]]
    }
    foreach entryName $entryNameList {
        prefs::removeObsolete [Bib::entryPrefConnect $entryName]
        lappend finalList $entryName
    }
    Bib::updatePreferences customEntry
    status::msg "\"$finalList\" fields have been restored to defaults."
}

# ===========================================================================
# 
# Remove (User-Defined) Custom Entry
# 
# Return the list of all (user defined) entryName preferences, and remove
# them.  These should be names of the form "myEntry", not "customMyEntry".
# 

proc Bib::removeCustomEntry {{entryNameList ""}} {
    
    global Bib::CustomEntryList1
    
    if {$entryNameList == ""} {
        set title "Select entries to remove :"
        set entryNameList [listpick -l -p $title [set Bib::CustomEntryList1]]
    }
    foreach entryName $entryNameList {
        prefs::removeObsolete [Bib::entryPrefConnect $entryName]
        lappend finalList $entryName
    }
    Bib::updatePreferences customEntry
    if {[set length [llength $finalList]] == 1} {
        status::msg "The custom entry \"$finalList\" has been removed."
    } else {
        status::msg "The custom entries \"$finalList\" have been removed."
    }
}

# ===========================================================================
# 
# Bib::addWindowFields
# 
# Add all of the "extra" fields which appear in entries in this window.
# 

proc Bib::addWindowFields {} {
    
    global win::Active Bib::DefaultFields BibmodeVars 
    
    if {${win::Active} == ""} {
	status::errorMsg "Cancelled -- no current window!"
    } 
    
    status::msg "Scanning [win::CurrentTail] for all fields"
    
    set length1 [llength $BibmodeVars(addFields)]
    set pos [minPos]
    set pat {^[\t ]*([a-zA-Z]+)[\t ]*=}
    set newFields ""
    while {![catch {search -s -f 1 -r 1 $pat $pos} match]} {
        set pos [nextLineStart [lindex $match 1]]
        set fieldLine [getText [lindex $match 0] [lindex $match 1]]
        regexp $pat [string tolower $fieldLine] match aField
        if {![lcontains Bib::DefaultFields $aField]} {
            append BibmodeVars(addFields) " $aField"
        } 
    }
    set BibmodeVars(addFields) [lsort [lunique $BibmodeVars(addFields)]]
    if {$length1 == [llength $BibmodeVars(addFields)]} {
        status::msg "No \"extra\" fields from this window were found."
        return -code return
    } else {
        prefs::modified BibmodeVars(addFields)
    }
    listpick -p "The custom fields include :" $BibmodeVars(addFields)
    status::msg ""
}

# ===========================================================================
# 
#  Acronyms  #
# 

proc Bib::acronymsProc {menuName item} {
    
    global BibmodeVars
    
    if {$menuName == "bibModeAcronyms" && [getModifiers]} {
        if {$item == "viewAcronyms"} {
            alertnote "Use this menu item to view the current set of\
              acronyms associated with Bib mode."
        } elseif {$item == "addAcronyms"} {
            alertnote "Use this menu item to add additional acronyms."
        } elseif {$item == "removeAcronyms"} {
            alertnote "Use this menu item to remove previously set acronyms."
        } elseif {$item == "unsetAcronymList"} {
            alertnote "Use this menu item to turn off the set of \
              pre-defined acronyms."
        } elseif {$item == "checkKeywords"} {
            alertnote "Use this menu item to check if a particular word\
              has already been defined as a Bib mode keyword."
        } elseif {$item == "bibModeTutorial"} {
            alertnote "Use this menu item to open a Bib mode\
              Completions Tutorial."
        } 
    } elseif {[regexp "setAcronymList" $item]} {
	set BibmodeVars($item) [expr $BibmodeVars($item) ? 0 : 1]
	prefs::modified BibmodeVars($item)
	# Anything else to do?
	if {[info exists flag::procs($item)]} {[set flag::procs($item)] $item} 
        Bib::unsetAcronymList $BibmodeVars(unsetAcronymList) 0
   } elseif {$item == "checkKeywords"} {
        Bib::checkKeywords
    } elseif {$item == "bibModeTutorial"} {
        mode::completionsTutorial "Bib"
    } else {
        Bib::$item
    } 
}

# ===========================================================================
# 
# View Acronyms
# 
# Place the names and elements of the array in a new window, and shrink it.
# 

proc Bib::viewAcronyms {{quietly 0}} {
    
    global Bib::Acronyms
    
    set windows [winNames]
    foreach w $windows {
        # Close any open "* Bib Acronyms *" windows.
        if {[regexp "\\* Bib Acronyms \\*" [win::StripCount $w]]} {
            bringToFront $w
            killWindow
            set quietly 0
        }
    }
    if {[set acroynymsList [listArray Bib::Acronyms]] == ""} {
        status::msg "There are currently no defined acronyms."
        return
    } elseif {$quietly} {
        return
    } 
    new -n "* Bib Acronyms *" -text $acroynymsList -m "Bib"
    # if 'shrinkWindow' is loaded, call it to trim the output window.
    catch {
        goto [maxPos] ; insertText "\r"
        selectAll     ; sortLines
    }
    goto [minPos]
    insertText "Command double-click on\racronyms to re-define them.\r\r"
    catch {shrinkWindow 2}
    winReadOnly
    status::msg ""
}

# ===========================================================================
# 
# Add Acronyms
# 
# Present the user with a dialog to create a new acronym.
# 

proc Bib::addAcronyms {{title ""} {acronym ""} {expansion ""}} {
    
    set finish [Bib::addAcronymsDialog $title $acronym $expansion]
    # Offer the dialog again to add more.
    set title "Create another acronym, or press Finish:"
    while {$finish != 1} {set finish [Bib::addAcronymsDialog $title]}
    Bib::viewAcronyms 1
    Bib::rebuildMenu bibModeAcronyms
}

proc Bib::addAcronymsDialog {{title ""} {acronym ""} {expansion ""}} {
    
    global Bib::Acronyms
    
    # Create the add acronym dialog.
    if {$title == ""} {set title "Create a new Acronym :"} 
    set y  10
    if {[info tclversion] < 8.0} {
        set aD [dialog::text $title 10 y 90]
        set yb 35
    } else {
        set aD [list -T $title]
        set yb 10
    }
    eval lappend aD [dialog::button    "Finish"                    290 yb   ]
    eval lappend aD [dialog::button    "More"                      290 yb   ]
    eval lappend aD [dialog::button    "Cancel"                    290 yb   ]
    if {$acronym == ""} {
        eval lappend aD \
          [dialog::textedit  "Acronym (3 letter max) :" $acronym    10  y  5]
    } else {
        eval lappend aD [dialog::text      "Acronym :"              10  y   ]
        eval lappend aD [dialog::menu 10 y $acronym $acronym                ]
    } 
    eval lappend aD     [dialog::textedit "Expansion:" $expansion   10  y 25]

    incr y 10
    set result    [eval dialog -w 370 -h $y $aD]
    set finish    [lindex $result 0]
    set cancel    [lindex $result 2]
    set acronym   [string trim [lindex $result 3]]
    set expansion [lindex $result 4]
    if {$cancel} {
        # User pressed "Cancel'
	status::errorMsg "Cancelled."
    } elseif {$acronym != "" || $expansion != ""} {
        set Bib::Acronyms($acronym) $expansion
        prefs::addArrayElement Bib::Acronyms $acronym $expansion
        status::msg "\"$acronym -- $expansion\" has been added."
        return $finish
    } elseif {$finish} {
        return $finish
    } else {
	status::errorMsg "Cancelled -- one of the required fields was empty."
    } 
}

# ===========================================================================
# 
# Edit Acronyms
# 
# Present the user with a dialog to edit a current misspelling.
# 

proc Bib::editAcronyms {{title ""} {acronym ""} {expansion ""}} {
    
    global Bib::Acronyms
    
    set acronyms [lsort [array names Bib::Acronyms]]
    if {$acronym == ""} {
        set acronym [listpick -p "Select an acronym to edit :" $acronyms] 
    } 
    if {$expansion == ""} {set expansion [set Bib::Acronyms($acronym)]} 
    if {$title == ""}     {set title "Edit the \"$acronym\" acronym:"} 
    set finish [Bib::addAcronymsDialog $title $acronym $expansion]
    # Offer the dialog again to add more.
    while {$finish != 1} {
        set title     "Select another acronym to edit, or Cancel :"
        set acronym   [listpick -p $title $acronyms]
        set expansion [set Bib::Acronyms($acronym)]
        set title     "Edit the \"$acronym\" acronym"
        set finish    [Bib::addAcronymsDialog $title $acronym $expansion]
    }
    Bib::viewAcronyms 1
}

# ===========================================================================
# 
# Remove Acronyms
# 
# Remove user-defined acronyms.  If "unsetList" is set, can alternatively
# remove the pre-defined list that is defined in "BibCompletions.tcl"
# Called from the menu, this will display the current list of acronyms at
# the end.
# 

proc Bib::removeAcronyms {{acronymList ""} {quietly 0}} {
    
    global Bib::Acronyms Bib::AcronymsSet
    
    if {$acronymList == ""} {
        # First list the user defined acronyms.
        set userAcronyms ""
        foreach acronym [array names Bib::Acronyms] {
            if {![info exists Bib::AcronymsSet($acronym)]} {
                # We know that this is user defined.
                lappend userAcronyms $acronym
            } elseif {[set Bib::Acronyms($acronym)] != [set Bib::AcronymsSet($acronym)]} {
                # We know that this has not been redefined.
                lappend userAcronyms $acronym
            } 
        } 
        if {![llength $userAcronyms]} {
            status::msg "Cancelled -- there are no user defined acronyms to remove."
            Bib::rebuildMenu bibModeAcronyms
            return
        } 
        set title "Select some acronyms to remove :"
        set acronymList [listpick -l -p $title [lsort $userAcronyms]]
    }
    
    # Remove them from "arrdefs.tcl"
    foreach acronym $acronymList {
        catch {prefs::removeArrayElement Bib::Acronyms $acronym}
        catch {unset Bib::Acronyms($acronym)}
    } 
    Bib::rebuildMenu bibModeAcronyms
    Bib::viewAcronyms $quietly
} 

# ===========================================================================
# 
# Unset Acronyms
# 
# Remove the pre-defined list of acronyms defined in "BibCompletions.tcl". 
# Called from the menu, this will change the value of the preference, and
# display the current list of acronyms at the end.
# 

proc Bib::unsetAcronymList {unsetList {quietly 1}} {
    
    global BibmodeVars Bib::Acronyms Bib::AcronymsSet
    
    if {!$unsetList} {
        # Restore the set of pre-defined acronyms
        foreach acronym [array names Bib::AcronymsSet] {
            if {![info exists Bib::Acronyms($acronym)]} {
                # We know that this has not been redefined.
                set Bib::Acronyms($acronym) [set Bib::AcronymsSet($acronym)]
            } 
        } 
        set BibmodeVars(unsetAcronymList) 0
    } else {
        # Remove the set of pre-defined acronyms
        foreach acronym [array names Bib::AcronymsSet] {
            if {[set Bib::Acronyms($acronym)] == [set Bib::AcronymsSet($acronym)]} {
                # We know that this has not been redefined.
                catch {unset Bib::Acronyms($acronym)}
            } 
        }
        set BibmodeVars(unsetAcronymList) 1
    }
    prefs::modified BibmodeVars(unsetAcronymList)
    Bib::viewAcronyms $quietly
}

# ===========================================================================
# 
# Check Keywords
# 
# See if the proposed custom entry / field name is already taken.
#

proc Bib::checkKeywords {{newKeywordList ""} {quietly 0} {noPrefs 0}} {
    
    global BibmodeVars Bib::DefaultEntries Bib::MyFldEntries Bib::DefaultFields 
    global Bib::CustomEntryList1 Bib::TeXCommandsList Bib::Acronyms Bib::Abbrevs
    
    set type 0
    if {$newKeywordList == ""} {
        set quietly 0
        set newKeywordList [prompt "Enter Bib mode keywords to be checked:" ""]
    }
    # Check to see if the new keyword(s) is already defined.
    foreach newKeyword $newKeywordList {
        if {[lsearch [set Bib::DefaultFields] $newKeyword] != "-1"} {
            set type "field"
        } elseif {[lsearch [set Bib::DefaultEntries] $newKeyword] != "-1"} {
            set type "entry"
        } elseif {[lsearch [set Bib::CustomEntryList1] $newKeyword] != "-1"} {
            set type "custom entry"
        } elseif {[lsearch [set Bib::MyFldEntries] $newKeyword] != "-1"} {
            set type "Bib::MyFlds() array"
        } elseif {[lsearch [set Bib::TeXCommandsList] $newKeyword] != "-1"} {
            set type "TeX Commands"
        } elseif {[lsearch [array names Bib::Acronyms] $newKeyword] != "-1"} {
            set type "Bib Acronyms"
        } elseif {[lsearch [set Bib::Abbrevs] $newKeyword] != "-1"} {
            set type "Standard Abbreviations"
        } elseif {!$noPrefs && \
          [lsearch $BibmodeVars(addFields) $newKeyword] != "-1"} {
            set type "Add Fields preference"
        }
        if {$quietly == 1} {
            # When this is called from other code, it should only contain
            # one keyword to be checked, and we'll return it's type.
            return "$type"
        } elseif {!$quietly && $type == 0} {
            alertnote "\"$newKeyword\" is not currently defined\
              as a Bib mode keyword"
        } elseif {$type != 0} {
            # This will work for any other value for "quietly", such as 2
            alertnote "\"$newKeyword\" is currently defined as a keyword\
              in the \"$type\" list."
        } 
        set type 0
    }
}

# ===========================================================================
# 
# .
